#ifndef SKSEARCH_H
#define SKSEARCH_H

#ifndef CFBASE_H
#include <CoreFoundation/CFBase.h>
#endif

#ifndef CFURL_H
#include <CoreFoundation/CFURL.h>
#endif

#ifndef CFDICTIONARY_H
#include <CoreFoundation/CFDictionary.h>
#endif

#ifndef SKINDEX_H
#include <SearchKit/SKIndex.h>
#endif

#include <AvailabilityMacros.h>

#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*SKSearchRef

Defines an opaque data type representing a an asynchronous search.

typedef struct __SKSearch *SKSearchRef;
Discussion

A search object is created when you call the SKSearchCreate function.
Availability
Available in Mac OS X v10.4 and later.
Declared In
SKSearch.h*/
typedef struct __SKSearch *SKSearchRef;

/*SKSearchOptions

Specifies the search options available for the SKSearchCreate function.

typedef UInt32 SKSearchOptions;
enum SKSearchType {
   kSKSearchOptionDefault = 0,
   kSKSearchOptionNoRelevanceScores = 1L << 0,
   kSKSearchOptionSpaceMeansOR = 1L << 1
   kSKSearchOptionFindSimilar = 1L << 2
};
Constants
kSKSearchOptionDefault

Default search options include:
Relevance scores will be computed
Spaces in a query are interpreted as Boolean AND operators.
Do not use similarity searching.

Available in Mac OS X v10.4 and later.
kSKSearchOptionNoRelevanceScores

This option saves time during a search by suppressing the computation of relevance scores.

Available in Mac OS X v10.4 and later.
kSKSearchOptionSpaceMeansOR

This option alters query behavior so that spaces are interpreted as Boolean OR operators.

Available in Mac OS X v10.4 and later.
kSKSearchOptionFindSimilar

This option alters query behavior so that Search Kit returns references to documents that are similar to an example text string. When this option is specified, Search Kit ignores all query operators.

Available in Mac OS X v10.4 and later.
Declared In
SKSearch.h*/
typedef UInt32 SKSearchOptions;
enum SKSearchType {
   kSKSearchOptionDefault = 0,
   kSKSearchOptionNoRelevanceScores = 1L << 0,
   kSKSearchOptionSpaceMeansOR = 1L << 1,
   kSKSearchOptionFindSimilar = 1L << 2
};


//function

/*SKSearchCreate

Creates an asynchronous search object for querying an index, and initiates search.

SKSearchRef SKSearchCreate (
   SKIndexRef              inIndex,
   CFStringRef             inQuery,
   SKSearchOptions         inSearchOptions
);
Parameters
inIndex

The index to query.
inQuery

The query string to search for.
inSearchOptions

The search options. May be NULL. See the SKSearchOptions enumeration for a description of the available options.
Return Value

A search object.
Discussion

This function creates an asynchronous search object for querying the document contents in an index. It also initiates the search on a separate thread.

After you create the search object, call SKSearchFindMatches to retrieve results. You can call SKSearchFindMatches immediately. To cancel a search, call SKSearchCancel.

For normal (non-similarity-based) queries, Search Kit discerns the type of query—Boolean, prefix, phrase, and so on—from the syntax of the query itself. Moreover, Search Kit supports multiple query types within a single search. For example, the following query includes Boolean, prefix, and suffix searching:appl* OR *ing


This query will return documents containing words that begin with “appl” as well as documents that contain words that end with “ing”.

For similarity searches, specified with the kSKSearchOptionFindSimilar flag in the inSearchOptions parameter, SKSearchCreate ignores all query operators.

The query operators that SKSearchCreate recognizes for non-similarity searching are:
Table 1  Search Kit query operators for non-similarity searches
Operator	
meaning

AND	
Boolean AND

&	
Boolean AND

<space>	
Boolean AND by default when no other operator is present, or Boolean OR if specified by kSKSearchOptionSpaceMeansOR.

OR	
Boolean inclusive OR

|	
Boolean inclusive OR

NOT	
Boolean NOT (see Special Considerations)

!	
Boolean NOT (see Special Considerations)

*	
Wildcard for prefix or suffix; surround term with wildcard characters for substring search. Ignored in phrase searching.

(	
Begin logical grouping

)	
End logical grouping

"	
delimiter for phrase searching


The operators AND, OR, and NOT are case sensitive.

Search Kit performs Unicode normalization on query strings and on the text placed into indexes. It uses Unicode Normalization Form KC (NFKC, compatibility decomposition followed by canonical composition) as documented in Unicode Standard Annex #15. For example, the a-grave character, ‘à’, can be written as the two Unicode characters (0x0061, 0x0300) or as the single Unicode character 0x00E0. Search Kit will normalize (0x0061, 0x0300) to 0x00E0. For more information on Unicode normalization, see http://unicode.org/reports/tr15.

Search Kit further normalizes query strings and indexes by stripping diacritical marks and by forcing characters to lowercase. For example, Search Kit normalizes each of the following characters to ‘a’: ‘a’, ‘à’, ‘A’, and ‘À’.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.

When your application no longer needs the search object, dispose of it by calling CFRelease.
Special Considerations

Search Kit supports logical exclusion. The NOT and ! operators behave as though they were EXCLUDE operators. For example, a search for ‘red NOT blue’ returns all documents that contain the word ‘red’ and do not contain the word ‘blue’.

Unary Boolean operators, however, are not currently implemented in Search Kit. A search, for example, for ‘NOT blue’, returns zero documents no matter what their content. 
Version Notes

Mac OS X version 10.4 uses a completely revised, and far more powerful, query approach than did earlier versions of Mac OS X. Refer to the Discussion in this function for details. Refer to SKSearchResultsCreateWithQuery (deprecated) for a description of Search Kit’s behavior in earlier versions of Mac OS X.

In versions of Mac OS X prior to version 10.4, Search Kit did not perform Unicode normalization, and did not remove diacritical marks.
Availability
Available in Mac OS X v10.4 and later.
Declared In
SKSearch.h*/
SKSearchRef SKSearchCreate (
   SKIndexRef              inIndex,
   CFStringRef             inQuery,
   SKSearchOptions         inSearchOptions
);

/*SKSearchFindMatches

Extracts search result information from a search object.

Boolean SKSearchFindMatches (
   SKSearchRef         inSearch,
   CFIndex             inMaximumCount,
   SKDocumentID        *outDocumentIDsArray,
   float               *outScoresArray,
   CFTimeInterval      maximumTime
   CFIndex             *outFoundCount
);
Parameters
inSearch

A reference to a search object (SKSearchRef) previously created with SKSearchCreate.
inMaximumCount

The maximum number of items to find. For each item found, SKSearchFindMatches places the associated document ID into the outDocumentIDsArray array. Specify an inMaximumCount of 0 to find as many items as possible within maximumTime.
outDocumentIDsArray

On input, a pointer to an array for document IDs. On output, points to points to the previously allocated array, which now contains the found document IDs. The size of this array must be equal to inMaximumCount.
outScoresArray

On input, a pointer to an array for scores. On output, points to the previously allocated array, which now contains relevance scores for the found items. The size of this array, if not NULL, must be equal to inMaximumCount. Can be NULL on input, provided that your application doesn’t need this information. Search Kit does not normalize relevance scores, so they can be very large.
maximumTime

The maximum number of seconds before this function returns, whether or not inMaximumCount items have been found. Setting maximumTime to 0 tells the search to return quickly
outFoundCount

On input, a pointer to a CFIndex object that will hold the number of items found. On output, points to the CFIndex object that now contains the actual number of items found.
Return Value

A logical value indicating whether the search is still in progress. Returns false when the search is exhausted.
Discussion

The SKSearchFindMatches extracts results from a find operation initiated by a search object (SKSearchRef).

This function provides results to its output parameters simply in the order in which they are found. This reduces latency to support search-as-you-type functionality. Larger scores mean greater relevance.

You can call this function on a search object repeatedly to get additional sets of search results. For example, if you call this function twice with an inMaximumCount value of 10, the first call will put the first 10 items found into the output arrays and the second call will put the second 10 items found into the output arrays.

Applications are free to determine relevance in any appropriate manner. One simple way is to divide each relevance score by the largest number returned to get relevance numbers scaled linearly from 0.0 to 1.0. Search Kit does not scale the relevance scores for you, because you may want to combine the scores from several calls on a search object or the scores from calls to more than one search object.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.

Before invoking a search, call SKIndexFlush on all indexes you will query to ensure that updates to the indexes have been flushed to disk.
Availability
Available in Mac OS X v10.4 and later.
Declared In
SKSearch.h*/
Boolean SKSearchFindMatches (
   SKSearchRef         inSearch,
   CFIndex             inMaximumCount,
   SKDocumentID        *outDocumentIDsArray,
   float               *outScoresArray,
   CFTimeInterval      maximumTime,
   CFIndex             *outFoundCount
);

/*SKIndexCopyInfoForDocumentIDs

Gets document names and parent IDs based on document IDs.

void SKIndexCopyInfoForDocumentIDs (
   SKIndexRef      inIndex,
   CFIndex         inCount,
   SKDocumentID    *inDocumentIDsArray,
   CFStringRef     *outNamesArray,
   SKDocumentID    *outParentIDsArray
);
Parameters
inIndex

The index containing the document information.
inCount

The number of document IDs in inDocumentIDsArray.
inDocumentIDsArray

Points to an array of document IDs representing the documents whose names and parent IDs you want.
outNamesArray

On input, a pointer to an array for document names. On output, points to the previously allocated array, which now contains the document names corresponding to the document IDs in inDocumentIDsArray. May be NULL on input if you don’t want to get the document names.

When finished with the names array, dispose of it by calling CFRelease on each array element.
outParentIDsArray

On input, a pointer to an array for parent document IDs. On output, points to the previously allocated array, which now contains document IDs representing the parents of the documents whose IDs are in inDocumentIDsArray. May be NULL on input if you don’t want to get the parent document IDs.
Discussion

The SKIndexCopyInfoForDocumentIDs function lets you get a batch of document names and parent document IDs in one step, based on a list of document IDs.
Availability
Available in Mac OS X v10.4 and later.
Declared In
SKSearch.h*/
void SKIndexCopyInfoForDocumentIDs (
   SKIndexRef      inIndex,
   CFIndex         inCount,
   SKDocumentID    *inDocumentIDsArray,
   CFStringRef     *outNamesArray,
   SKDocumentID    *outParentIDsArray
);


#ifdef __cplusplus
}
#endif

#endif