/*
 * $Id: setup.c,v 1.1.1.1 2002/09/19 00:37:09 halite Exp $
 *
 * Created by Halite on 2002.08.19
 */

#include <toykernel/init.h>
#include <toykernel/kernel.h>
#include <toykernel/string.h>
#include <toykernel/mm.h>
#include <toykernel/sched.h>
#include <arch/setup.h>
#include <arch/page.h>
#include <arch/pgtable.h>
#include <arch/e820.h>
#include <arch/processor.h>
#include <arch/desc.h>
#include <arch/current.h>

/*
 * section boundaries
 */
extern char _text, _etext, _edata, _end;

/* cpu information */
struct cpuinfo_x86 boot_cpu_data = { 0, 0, 0, 0, -1, 1, 0, 0, -1 };
/* memory region information */
struct e820map e820;

/*
 * This is set up by the setup-routine at boot-time
 */
#define PARAM		((unsigned char *)empty_zero_page)
#define E820_MAP	((struct e820entry *) (PARAM+0))
#define E820_MAP_NR	(*(unsigned short *) (PARAM+0x280))
#define E801_MEM	(*(unsigned long *) (PARAM+0x284))
#define MEM_88		(*(unsigned long *) (PARAM+0x288))

static void add_memory_region(unsigned long long start, unsigned long long size, unsigned long type)
{
	int x = e820.nr_map;

	if (x == E820MAX)
	{
		printk("Too many entries in the memory map!\n");
		return;
	}

	e820.map[x].addr = start;
	e820.map[x].size = size;
	e820.map[x].type = type;
	e820.nr_map++;
}

#define E820_DEBUG	1

static void __init print_memory_map(char *who)
{
	int i;

	for (i = 0; i < e820.nr_map; i++) {
		printk(" %s: %016Lx - %016Lx ", who,
			e820.map[i].addr,
			e820.map[i].addr + e820.map[i].size);
		switch (e820.map[i].type) {
		case E820_RAM:	printk("(usable)\n");
				break;
		case E820_RESERVED:
				printk("(reserved)\n");
				break;
		case E820_ACPI:
				printk("(ACPI data)\n");
				break;
		case E820_NVS:
				printk("(ACPI NVS)\n");
				break;
		default:	printk("type %lu\n", e820.map[i].type);
				break;
		}
	}
}

static int __init sanitize_e820_map(struct e820entry * biosmap, char * pnr_map)
{
	struct change_member {
		struct e820entry *pbios; /* pointer to original bios entry */
		unsigned long long addr; /* address for this change point */
	};
	struct change_member change_point_list[2*E820MAX];
	struct change_member *change_point[2*E820MAX];
	struct e820entry *overlap_list[E820MAX];
	struct e820entry new_bios[E820MAX];
	struct change_member *change_tmp;
	unsigned long current_type, last_type;
	unsigned long long last_addr;
	int chgidx, still_changing;
	int overlap_entries;
	int new_bios_entry;
	int old_nr, new_nr;
	int i;

	/*
		Visually we're performing the following (1,2,3,4 = memory types)...

		Sample memory map (w/overlaps):
		   ____22__________________
		   ______________________4_
		   ____1111________________
		   _44_____________________
		   11111111________________
		   ____________________33__
		   ___________44___________
		   __________33333_________
		   ______________22________
		   ___________________2222_
		   _________111111111______
		   _____________________11_
		   _________________4______

		Sanitized equivalent (no overlap):
		   1_______________________
		   _44_____________________
		   ___1____________________
		   ____22__________________
		   ______11________________
		   _________1______________
		   __________3_____________
		   ___________44___________
		   _____________33_________
		   _______________2________
		   ________________1_______
		   _________________4______
		   ___________________2____
		   ____________________33__
		   ______________________4_
	*/

	/* if there's only one memory region, don't bother */
	if (*pnr_map < 2)
		return -1;

	old_nr = *pnr_map;

	/* bail out if we find any unreasonable addresses in bios map */
	for (i=0; i<old_nr; i++)
		if (biosmap[i].addr + biosmap[i].size < biosmap[i].addr)
			return -1;

	/* create pointers for initial change-point information (for sorting) */
	for (i=0; i < 2*old_nr; i++)
		change_point[i] = &change_point_list[i];

	/* record all known change-points (starting and ending addresses) */
	chgidx = 0;
	for (i=0; i < old_nr; i++)	{
		change_point[chgidx]->addr = biosmap[i].addr;
		change_point[chgidx++]->pbios = &biosmap[i];
		change_point[chgidx]->addr = biosmap[i].addr + biosmap[i].size;
		change_point[chgidx++]->pbios = &biosmap[i];
	}

	/* sort change-point list by memory addresses (low -> high) */
	still_changing = 1;
	while (still_changing)	{
		still_changing = 0;
		for (i=1; i < 2*old_nr; i++)  {
			/* if <current_addr> > <last_addr>, swap */
			/* or, if current=<start_addr> & last=<end_addr>, swap */
			if ((change_point[i]->addr < change_point[i-1]->addr) ||
				((change_point[i]->addr == change_point[i-1]->addr) &&
				 (change_point[i]->addr == change_point[i]->pbios->addr) &&
				 (change_point[i-1]->addr != change_point[i-1]->pbios->addr))
			   )
			{
				change_tmp = change_point[i];
				change_point[i] = change_point[i-1];
				change_point[i-1] = change_tmp;
				still_changing=1;
			}
		}
	}

	/* create a new bios memory map, removing overlaps */
	overlap_entries=0;	 /* number of entries in the overlap table */
	new_bios_entry=0;	 /* index for creating new bios map entries */
	last_type = 0;		 /* start with undefined memory type */
	last_addr = 0;		 /* start with 0 as last starting address */
	/* loop through change-points, determining affect on the new bios map */
	for (chgidx=0; chgidx < 2*old_nr; chgidx++)
	{
		/* keep track of all overlapping bios entries */
		if (change_point[chgidx]->addr == change_point[chgidx]->pbios->addr)
		{
			/* add map entry to overlap list (> 1 entry implies an overlap) */
			overlap_list[overlap_entries++]=change_point[chgidx]->pbios;
		}
		else
		{
			/* remove entry from list (order independent, so swap with last) */
			for (i=0; i<overlap_entries; i++)
			{
				if (overlap_list[i] == change_point[chgidx]->pbios)
					overlap_list[i] = overlap_list[overlap_entries-1];
			}
			overlap_entries--;
		}
		/* if there are overlapping entries, decide which "type" to use */
		/* (larger value takes precedence -- 1=usable, 2,3,4,4+=unusable) */
		current_type = 0;
		for (i=0; i<overlap_entries; i++)
			if (overlap_list[i]->type > current_type)
				current_type = overlap_list[i]->type;
		/* continue building up new bios map based on this information */
		if (current_type != last_type)	{
			if (last_type != 0)	 {
				new_bios[new_bios_entry].size =
					change_point[chgidx]->addr - last_addr;
				/* move forward only if the new size was non-zero */
				if (new_bios[new_bios_entry].size != 0)
					if (++new_bios_entry >= E820MAX)
						break; 	/* no more space left for new bios entries */
			}
			if (current_type != 0)	{
				new_bios[new_bios_entry].addr = change_point[chgidx]->addr;
				new_bios[new_bios_entry].type = current_type;
				last_addr=change_point[chgidx]->addr;
			}
			last_type = current_type;
		}
	}
	new_nr = new_bios_entry;   /* retain count for new bios entries */

	/* copy new bios mapping into original location */
	memcpy(biosmap, new_bios, new_nr*sizeof(struct e820entry));
	*pnr_map = new_nr;

	return 0;
}

static int __init copy_e820_memory_info(struct e820entry *biosmap, int nr_map)
{
	if (nr_map < 2)
		return -1;

	do {
		unsigned long long start = biosmap->addr;
		unsigned long long size  = biosmap->size;
		unsigned long long end   = start + size;
		unsigned long      type  = biosmap->type;

		if (start > end)
			return -1;

		if (type == E820_RAM)
		{
			if (start < 0x100000ULL && end > 0xa0000ULL)
			{
				if (start < 0xa0000ULL)
					add_memory_region(start, 0xa0000ULL - start, type);
				if (end <= 0x100000ULL)
					continue;
				start = 0x100000ULL;
				size = end - start;
			}
		}
		add_memory_region(start, size, type);
	} while (biosmap++, --nr_map);

	return 0;
}

#define LOWMEMSIZE()	(0x9f000)

static void __init setup_memory_region(void)
{
	char *who = "BIOS-e820";

	sanitize_e820_map(E820_MAP, (char *)(&E820_MAP_NR));
	if (copy_e820_memory_info(E820_MAP, E820_MAP_NR) < 0)
	{
		unsigned long mem_size;

		/* compare results from other methods and take the greater */
		if (E801_MEM < MEM_88) {
			mem_size = MEM_88;
			who = "BIOS-88";
		} else {
			mem_size = E801_MEM;
			who = "BIOS-e801";
		}

		e820.nr_map = 0;
		add_memory_region(0, LOWMEMSIZE(), E820_RAM);
		add_memory_region(HIGH_MEMORY, mem_size << 10, E820_RAM);
	}
	printk("BIOS-provided physical RAM map:\n");
	print_memory_map(who);
}

#define PFN_UP(x)	(((x) + PAGE_SIZE-1) >> PAGE_SHIFT)
#define PFN_DOWN(x)	((x) >> PAGE_SHIFT)
#define PFN_PHYS(x)	((x) << PAGE_SHIFT)
#define MAXMEM_PFN	PFN_DOWN(MAXMEM)
void __init setup_arch(char **cmdline)
{
	unsigned long /*start_pfn,*/ max_pfn;
	int i;

	/* setup memory region from e820 function */
	setup_memory_region();
	/* get start page frame number after kernel in ram */
	min_low_pfn = PFN_UP(__pa(&_end));
	/* get maximum page frame number */
	max_pfn = 0;
	for (i = 0; i < e820.nr_map; i++)
	{
		unsigned long start, end;
		/* RAM? */
		if (e820.map[i].type != E820_RAM)
			continue;
		start = PFN_UP(e820.map[i].addr);
		end = PFN_DOWN(e820.map[i].addr + e820.map[i].size);
		if (start >= end)
			continue;
		if (end > max_pfn)
			max_pfn = end;
	}
	max_low_pfn = max_pfn;
	if (max_low_pfn > MAXMEM_PFN)
		max_low_pfn = MAXMEM_PFN;
}

void __init cpu_init(void)
{
	int nr = 0; //smp_processor_id();
	struct tss_struct *t = &init_tss[nr];

	/* initialize tss */

	__asm__ ("lgdt %0":"=m" (gdt_descr));
	__asm__ ("lidt %0":"=m" (idt_descr));

	/* delete NT */
	__asm__ ("pushfl; andl $0xffffbfff, (%esp); popfl");

	/*
	 * set up and load the per-CPU TSS and LDT
	 */
#if 0
	atomic_inc(&init_mm.mm_count);
	current->active_mm = &init_mm;
	if(current->mm)
		BUG();
	enter_lazy_tlb(&init_mm, current, nr);
#endif

	/*
	 * head.S %esp init_task_union+8192 Ƿ
	 * current init_task Ű ȴ.
	 */
	t->esp0 = current->thread.esp0;
	set_tss_desc(nr, t);
	gdt_table[__TSS(nr)].b &= 0xfffffdff;
	load_TR(nr);
//	load_LDT(&init_mm);

	/*
	 * Clear all 6 debug registers:
	 */

#define CD(register) __asm__("movl %0,%%db" #register ::"r"(0) );

	CD(0); CD(1); CD(2); CD(3); /* no db4 and db5 */; CD(6); CD(7);

#undef CD

#if 0
	/*
	 * Force FPU initialization:
	 */
	current->flags &= ~PF_USEDFPU;
	current->used_math = 0;
	stts();
#endif
}

