/*
 * $Id$
 */

#include <toykernel/kernel.h>
#include <toykernel/mm.h>
#include <toykernel/init.h>

// described in toykernel.lds
extern char _text, _end;

unsigned long max_low_pfn;
unsigned long min_low_pfn;
mem_map_t *mem_map;

static struct page *find_free_page(void)
{
	mem_map_t *p;
	int i;

	for (i = 0, p = mem_map; i < max_low_pfn && p->count; i++, p++)
		continue;
	if (i >= max_low_pfn || p->count)
		return 0;
	return p;
}

struct page * __alloc_pages(unsigned int gfp_mask, unsigned int size)
{
	unsigned int final_nr, nr_got;
	mem_map_t *start, *end, *p, *n;

	final_nr = (size + PAGE_SIZE - 1) >> PAGE_SHIFT;
	start = find_free_page();
	if (!start)
		return 0;

	start->count = 1;
	start->prev  = (mem_map_t *)NULL;
	start->next  = (mem_map_t *)NULL;

	end    = mem_map + max_low_pfn;
	nr_got = 1;
	p      = start;
	n      = p + 1;
	while (nr_got < final_nr && n < end)
	{
		// find another
		if (n->count)
		{
			n = find_free_page();
			if (!n)
			{
				printk("Memory: not enough memory to allocate.\n");
				goto alloc_error;
			}
		}
		// assign
		n->count = 1;
		p->next  = n;
		n->prev  = p;
		n->next  = (mem_map_t *)NULL;
		p        = n;
		n        = p + 1;
		nr_got++;
	}

	return start;

alloc_error:
	while (start && start->next)
	{
		start->count = 0;
		start->prev  = (mem_map_t *)NULL;
		n            = start->next;
		start->next  = (mem_map_t *)NULL;
		start        = n;
	}

	return 0;
}

unsigned long __get_free_pages(unsigned int gfp_mask, unsigned int size)
{
	struct page *page;

	page = alloc_pages(gfp_mask, size);
	if (!page)
		return 0;
	return (unsigned long)page_address(page);
}

unsigned long get_zeroed_page(unsigned int gfp_mask)
{
	struct page * page;

	page = alloc_pages(gfp_mask, PAGE_SIZE);
	if (page) {
		void *address = page_address(page);
		clear_page(address);
		return (unsigned long) address;
	}
	return 0;
}

static void FASTCALL(__free_pages_ok (struct page *page, unsigned int order));
static void __free_pages_ok(struct page *page, unsigned int order)
{
	mem_map_t *n;

	while (page && page->next)
	{
		page->count = 0;
		page->prev  = (mem_map_t *)NULL;
		n           = page->next;
		page->next  = (mem_map_t *)NULL;
		page        = n;
	}
}

void __free_pages(struct page *page, unsigned int order)
{
	//if (!PageReserved(page) && put_page_testzero(page))
		__free_pages_ok(page, order);
}

void free_pages(unsigned long addr, unsigned int order)
{
	if (addr != 0)
		__free_pages(virt_to_page(addr), order);
}

/*
 * Initialize mem_map array
 */
extern void reserve_mem_map(mem_map_t *mmap);

int __init mem_map_init(void)
{
	mem_map_t *p;
	unsigned long start_mem;

	/* mem_map starts from just after the kernel. */
	mem_map   = (mem_map_t *) &_end;
	p         = mem_map + max_low_pfn;
	start_mem = ((unsigned long)p + sizeof(long) - 1) & ~(sizeof(long) - 1);
	memset(mem_map, 0, start_mem - (unsigned long)mem_map);
	/* top to bottom clear */
	do {
		--p;
		p->count = 0;
	} while (p > mem_map);

	return 0;
}

